//------------------------------------------------------------------------------
// File: OutpostGold.cs
// Handles gold trade outpost mission
// Author: Michael Felice
//------------------------------------------------------------------------------


// SCENARIO CALLBACK FUNCTIONS for Outpost Gold Mission

// load scenario function (initializes the outpost gold mission tasks)
function startOutpostGoldTasks()
{
   tsStartTask("OutpostGold", "OutpostGoldInit");
   
   // disable the train station upgrade
   tsEnableCommand(TrainStationUpgradeCmdData, false);
   
   // disable trains from moving on this map
   slgTrain_DisableTrains();
   
   // disable the quit scenario button
   $DisableQuitScenario = true;
}

// quit scenario function (turns off the outpost gold task)
function quitOutpostGoldTasks()
{
   tsEndTask("OutpostGold");
   
   // disable gold production
   tsEnableCommand(ProduceGoldCmdData, false);
   
   // check if the building should be unselected
   if (csIsCommandDisabled(ProduceFoodCmdData) == true &&
      csIsCommandDisabled(ProduceWoodCmdData) == true)
   {
      %object = nameToId("mapplatform");
      %building = slgQueryInterface(%object, $CID_BUILDING);
      %building.startExclusiveProduce($Resource::Count);
   }
   
   // reset the gold trigger
   TaskModifier.strResetVertex("OutpostTasks", "OutpostTaskGoldTrigger");
   TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskGoldTrigger", $TSV_AVAIL);
   
   // resend the alert message
   SendOutpostTaskAlert("ID_TASKNAME_OUTPOSTGOLD", "id_outpost_gold");
}



// MAIN SET OF TASKS

// explain the goal of the map, set up quests
function OutpostGoldInit::onExit(%vertex)
{
   // send the information
   tsSendTaskPopUpDlg("ID_TASKNAME_OUTPOSTGOLD", "id_outpostgold_intro",
      $TaskButton::Okay, HeroCharacter, "id_outpostgold_intro");
   
   // set up the quest
   tsAddQuestToTracker(true, "Outpost Gold Mission",
      "Talk to " @ HeroCharacter.name @ " to find out what we need to do before we can import Gold at our town's Outpost.",
      "Before you can start receiving shipments of Gold back at town, you will need to establish a source for your town's additional Gold supply.",
      "Reward: Import Gold at your Outpost and 10 Experience");
   tsReadyQuest("Outpost Gold Mission");
}

// tell the player how to start the quest
function OutpostGoldStart::onExit(%vertex)
{
   // update the base quest information
   tsAddTaskToTracker("Outpost Gold Mission", "Establish a source of additional Gold.",
      "Complete all of the necessary tasks to establish a source of additional Gold for your town.");
   
   // add the new task for building the goldmines
   tsAddTaskToTracker("Outpost Gold Mission", "Build three Gold Mines.",
      "Build three Gold Mines in order establish a source of additional Gold for your town.");
}



// HANDLES BUILDING AND DESTROYING THREE GOLD MINES

// this occurs when one gold mine has been created
function OutpostGoldGMCreate1::onExit(%vertex)
{
}

// this occurs when two gold mines have been created
function OutpostGoldGMCreate2::onExit(%vertex)
{
}

// this occurs when three gold mines have been created
function OutpostGoldGMCreate3::onExit(%vertex)
{
   tsCompleteTaskInTracker("Outpost Gold Mission", "Build three Gold Mines.");
   
   // check if the gold outpost should be started
   if (OutpostGoldOPStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPStart1", $TSV_CMPLETE);
   }
   
   // check if the the gold resource collection should start
   if (OutpostGoldOPCreate.isMarkedComplete() == true &&
      OutpostGoldResourceStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceStart1", $TSV_CMPLETE);
   }
   
   // check if everything is complete
   if (OutpostGoldOPCreate.isMarkedComplete() == true &&
      OutpostGoldResourceCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldComplete1", $TSV_AVAIL);
   }
}

// this occurs when a gold mine is destroyed
function OutpostGoldGMDestroy::onMessage(%vertex, %type, %param)
{
   // check if a gold mine is destroyed, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was no affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMDestroy", $TSV_AVAIL);
   
   // find out how many active gold mines are remaining
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   %goldmineCount = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // get the object in the list
      %object = %list.getId(%index);
      
      // if the object datablock is not a gold mine, move to the next object
      %datablock = %object.getDataBlock();
      if (%datablock.getName() !$= "Goldmine")
      {
         continue;
      }
      
      // if the gold mine is in the productino state, add to the gold mine count
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %goldmineCount++;
      }
   }
   
   // update the vertices based on how many gold mines are available
   switch (%goldmineCount)
   {
      case 0:
         // no gold mines are active
         TaskModifier.strResetVertex("OutpostGold", "OutpostGoldGMCreate1");
         TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMCreate1", $TSV_AVAIL);

      case 1:
         // one gold mine is active
         TaskModifier.strResetVertex("OutpostGold", "OutpostGoldGMCreate2");
         TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMCreate2", $TSV_AVAIL);

      case 2:
         // two gold mines are active (lose the gold mine completion message)
         TaskModifier.strResetVertex("OutpostGold", "OutpostGoldGMCreate3");
         TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMCreate3", $TSV_AVAIL);
         tsAddTaskToTracker("Outpost Gold Mission", "Build three Gold Mines.",
            "Build three Gold Mines in order establish a source of additional Gold for your town.");
   }
}



// when the previous conditions are met for the first time, and the
// platform needs to be upgraded
function OutpostGoldOPStart1::onExit(%vertex)
{
   // give the hero something to talk about
   tsSendTaskClickDlg("ID_TASKNAME_OUTPOSTGOLD", "id_outpostgold_platform",
      $TaskButton::Okay, HeroCharacter, "id_outpostgold_platform");
}

// set up the platform for the upgrade
function OutpostGoldOPStart2::onExit(%vertex)
{
   // update the quest information
   tsAddTaskToTracker("Outpost Gold Mission", "Upgrade the Platform to an Outpost.",
      "Upgrade the Platform to an Outpost in order to ship Gold back to town.");
}

// before the outpost can be created, the platform team needs to be set
function OutpostGoldOPCreate::onEnter(%vertex)
{
   // update the team of the platform
   resourcePlatform.setTeam($OST_PLAYER);
}

// this occurs when the outpost has been upgraded
function OutpostGoldOPCreate::onExit(%vertex)
{
   tsCompleteTaskInTracker("Outpost Gold Mission", "Upgrade the Platform to an Outpost.");
   
   // check if the gold mines are also completed
   if (OutpostGoldGMCreate3.isMarkedComplete() == true &&
      OutpostGoldResourceStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceStart1", $TSV_CMPLETE);
   }
   
   // check if all of the tasks have been completed
   if (OutpostGoldGMCreate3.isMarkedComplete() == true &&
      OutpostGoldResourceCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldComplete1", $TSV_AVAIL);
   }
}

// this occurs when the outpost is destroyed
function OutpostGoldOPDestroy::onMessage(%vertex, %type, %param)
{
   // check if the outpost is destroyed, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was no affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPDestroy", $TSV_AVAIL);
   
   // if the outpost is lost, we know it is the player's outpost because
   // it is the only outpost on the map, so reactivate the outpost task
   TaskModifier.strResetVertex("OutpostGold", "OutpostGoldOPCreate");
   TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPCreate", $TSV_AVAIL);
   tsAddTaskToTracker("Outpost Gold Mission", "Upgrade the Platform to an Outpost.",
      "Upgrade the Platform to an Outpost in order to ship Gold back to town.");
}



// HANDLES RESOURCE COLLECTION

// after the gold mines and outpost have been created, send
// a message for resource collection
function OutpostGoldResourceStart1::onExit(%vertex)
{
   // give the hero something to talk about
   tsSendTaskClickDlg("ID_TASKNAME_OUTPOSTGOLD", "id_outpostgold_resource",
      $TaskButton::Okay, HeroCharacter, "id_outpostgold_resource");
}

// set up messages for resource collection
function OutpostGoldResourceStart2::onExit(%vertex)
{
   // update the quest information
   tsAddTaskToTracker("Outpost Gold Mission", "Collect 500 Gold.",
      "A stockpile of 500 Gold should be enough to begin sending Gold back to town.");
}

// check if gold resources have passed 500
function OutpostGoldResourceCreate::onUpdate(%vertex)
{
   // if not enough gold has been collected, do nothing
   if (rsGetResourceCount($Resource::Gold) < 500)
   {
      return;
   }
   
   // complete this step of resource collection
   TaskModifier.StrMarkVertex("OutpostGold", "OutpostGoldResourceCreate", $TSV_CMPLETE);
   TaskModifier.StrMarkVertex("OutpostGold", "OutpostGoldResourceDestroy", $TSV_AVAIL);
   
   // update the task to completion
   tsCompleteTaskInTracker("Outpost Gold Mission", "Collect 500 Gold.",
      "A stockpile of 500 Gold should be enough to begin sending Gold back to town.");
   
   // if all of the tasks are completed, we are done
   if (OutpostGoldGMCreate3.isMarkedComplete() == true &&
      OutpostGoldOPCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldGMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostGold", "OutpostGoldComplete1", $TSV_AVAIL);
   }
}

// check if gold resources have falled below 500
function OutpostGoldResourceDestroy::onUpdate(%vertex)
{
   // if enough gold has been collected, do nothing
   if (rsGetResourceCount($Resource::Gold) >= 500)
   {
      return;
   }
   
   // reactivate this step of resource collection
   TaskModifier.StrMarkVertex("OutpostGold", "OutpostGoldResourceCreate", $TSV_AVAIL);
   TaskModifier.StrMarkVertex("OutpostGold", "OutpostGoldResourceDestroy", $TSV_CMPLETE);
   
   // update the task to incompletion
   tsAddTaskToTracker("Outpost Gold Mission", "Collect 500 Gold.",
      "A stockpile of 500 Gold should be enough to begin sending Gold back to town.");
}



// HANDLES OUTPOST GOLD COMPLETION

// notify the player that mission is done
function OutpostGoldComplete1::onEnter(%vertex)
{
   // give the hero something to talk about
   tsSendTaskPopUpDlg("ID_TASKNAME_OUTPOSTGOLD", "id_outpostgold_complete",
      $TaskButton::Okay, HeroCharacter, "id_outpostgold_complete");
}

// close the quests that have been completed and update experience
function OutpostGoldComplete1::onExit(%vertex)
{
   // complete the quest
   tsCompleteTaskInTracker("Outpost Gold Mission", "Establish a source of additional Gold.");
   
   // experience reward for completing the quest
   tsModifyResource($Resource::Experience, 10, true);
   SendProductionToClient(HeroCharacter, "10 experience");
}

// exit the satellite
function OutpostGoldComplete2::onExit(%vertex)
{
   // return from this satellite
   %scenario = new SLScenario()
   {
        stack = false;
   };
   LoadScenario(%scenario);
   
   // update the satellite flag
   mapsatellite_outpostgold.disabled = false;
   TaskModifier.strMarkVertex("Badgetasks", "OutpostTaskGoldComplete", $TSV_CMPLETE);
}
